# ---
# jupyter:
#   jupytext:
#     text_representation:
#       extension: .py
#       format_name: percent
#       format_version: '1.3'
#       jupytext_version: 1.18.1
#   kernelspec:
#     display_name: gwpy4
#     language: python
#     name: gwpy4
# ---

# %%
import os
import sys

# %%
import matplotlib as mpl
import numpy as np

from matplotlib import pyplot as plt

# %%
import warnings
warnings.filterwarnings("ignore", "Wswiglal-redir-stdio")

from gwpy import timeseries as gwts

# %%
mpl.rc_file_defaults()
mpl.rcParams.update({
    "figure.dpi": 200,
    "figure.figsize": (3.75, 3),
    "font.family": "serif",
    "savefig.bbox": "tight",
    "savefig.pad_inches": 0.05,
    "text.usetex": True,
})


# %%
def makegrid(ax):
    ax.grid(True, which="major", color="k", alpha=0.05)
    ax.grid(True, which="minor", color="k", alpha=0.025)


# %%
# From T2200274
temps = np.array([18.7, 28.4, 34.7, 47.1, 55.6])
rocs_h = np.array([2.141, 2.017, 1.947, 1.821, 1.743])
rocs_v = np.array([2.135, 2.004, 1.944, 1.820, 1.742])
roc_from_temp = np.polyfit(temps, rocs_v, 2)

# %%
fig_om2, ax_om2 = plt.subplots()
ax_om2.plot(temps, rocs_h, 'o', fillstyle="none", label="Sagittal", color="xkcd:hot pink")
ax_om2.plot(temps, rocs_v, 'o', fillstyle="none", label="Tangential", color="xkcd:forest green")
temps_fine = np.linspace(15, 65, 100)
ax_om2.plot(temps_fine, np.polyval(roc_from_temp, temps_fine), label="Fit (tang.)", c="xkcd:green")
ax_om2.set_xlabel("Temperature [$^\circ$C]")
ax_om2.set_ylabel("Radius of curvature [m]")
ax_om2.set_title("Characterization from LIGO--T2200274")
ax_om2.legend()
makegrid(ax_om2)
fig_om2.savefig("om2_temp_RoC_T2200274.pdf")

# %%
cal_chan = "H1:CAL-CS_TDEP_KAPPA_C_OUTPUT"
temp_chan = "H1:AWC-OM2_TSAMS_THERMISTOR_2_TEMPERATURE"
pwr_chan = "H1:ASC-X_PWR_CIRC_OUT16"
channels = [cal_chan, temp_chan, pwr_chan]

# %%
fnames = ["h1_om2_test_1.h5", "h1_om2_test_2.h5"]
times = [
    ("2023-06-27 11:45:00 Z", "2023-06-27 13:30:00 Z"),
    ("2025-09-04 22:00:00 Z", "2025-09-04 22:45:00 Z"),
]
data_dicts = []

# %%
for fname, (tstart, tstop) in zip(fnames, times):
    if os.path.exists(fname):
        data = gwts.TimeSeriesDict.read(fname)
        print(f"Read {fname} from local disk.")
    else:
        data = gwts.TimeSeriesDict.fetch(channels, tstart, tstop, allow_tape=True, verbose=True, host="nds.ligo-wa.caltech.edu")
        data.write(fname)
    data_dicts.append(data)

# %%
fig_kc, ax_kc = plt.subplots()
for data, (tstart, tstop) in zip(data_dicts, times):
    mask = data[temp_chan].value > 25.5 # to chop off some transients
    args = np.argsort(data[temp_chan][mask].value)
    ax_kc.plot(np.polyval(roc_from_temp, data[temp_chan][mask][args].value), data[cal_chan][mask][args], label=f"{tstart} to {tstop[10:]}")
ax_kc.set_xlabel("OM2 RoC [m]")
ax_kc.set_ylabel(r"Normalized optical gain $\kappa_\mathrm{c}$")
ax_kc.legend(fontsize="small")
ax_kc.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.05))
ax_kc.set_ylim([0.975, 1])
ax_kc.set_title("OM2 tests at Hanford in O4")
makegrid(ax_kc)
fig_kc.savefig("h1_om2_kc_o4.pdf")

# %%
